<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class UserRoleController extends Controller
{
    public function index(Request $request)
    {
        if ($request->has('dashboard')) {
            $superAdmins = User::whereHas('role', function ($query) {
                $query->where('name', 'super-admin');
            })->limit(3)->get();

            $users = User::whereHas('role', function ($query) {
                $query->where('name', 'user');
            })->limit(3)->get();

            $otherRoles = User::whereHas('role', function ($query) {
                $query->whereNotIn('name', ['super-admin', 'user']);
            })->limit(3)->get();

            return response()->json([
                'super-admins' => $superAdmins,
                'users' => $users,
                'other-roles' => $otherRoles,
            ]);
        }
        
        $query = User::with('role');

        // Filter by role type. Admins are users whose role is not 'User'.
        if ($request->input('role_type') === 'admin') {
            $query->whereHas('role', function ($q) {
                $q->where('name', '!=', 'User');
            });
        } else {
            // Default to fetching standard users
            $query->whereHas('role', function ($q) {
                $q->where('name', 'User');
            });
        }

        $users = $query->get();

        return response()->json($users);
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8',
            'role_id' => 'required|exists:roles,id',
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'role_id' => $request->role_id,
        ]);

        return response()->json($user, 201);
    }

    public function show(User $user)
    {
        return response()->json($user->load('role'));
    }

    public function update(Request $request, User $user)
    {
        $request->validate([
            'name' => 'string|max:255',
            'email' => 'string|email|max:255|unique:users,email,' . $user->id,
            'role_id' => 'exists:roles,id',
        ]);

        $user->update($request->all());

        return response()->json($user);
    }

    public function destroy(User $user)
    {
        $user->delete();

        return response()->json(null, 204);
    }
}
