<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Validation\Rule;
use App\Models\Permission;

class RoleController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return Cache::remember('roles', 60 * 5, function () {
            $roles = Role::with('permissions')->get();
            $superAdmin = $roles->firstWhere('name', 'Super Admin');
            if ($superAdmin) {
                $superAdmin->permissions = Permission::all();
            }
            return $roles;
        });
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|unique:roles,name|max:255',
            'permissions' => 'nullable|array',
            'permissions.*' => 'integer|exists:permissions,id',
        ]);

        $role = Role::create(['name' => $validated['name']]);

        if (!empty($validated['permissions'])) {
            $role->permissions()->sync($validated['permissions']);
        }

        Cache::forget('roles');

        return response()->json($role->load('permissions'), 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(Role $role)
    {
        if ($role->name === 'Super Admin') {
            $role->permissions = Permission::all();
        }
        return $role->load('permissions');
    }

    /**
     * Update the specified resource's name in storage.
     */
    public function update(Request $request, Role $role)
    {
        if ($role->name === 'Super Admin') {
            return response()->json(['message' => 'The Super Admin role cannot be updated.'], 403);
        }

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255', Rule::unique('roles')->ignore($role->id)],
        ]);

        $role->update($validated);

        Cache::forget('roles');

        return $role->load('permissions');
    }

    /**
     * Update the permissions for a specific role.
     */
    public function updatePermissions(Request $request, Role $role)
    {
        if ($role->name === 'Super Admin') {
            return response()->json(['message' => 'The Super Admin role permissions cannot be updated.'], 403);
        }

        $validated = $request->validate([
            'permissions' => 'nullable|array',
            'permissions.*' => 'integer|exists:permissions,id',
        ]);

        $role->permissions()->sync($validated['permissions'] ?? []);

        Cache::forget('roles');

        return $role->load('permissions');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Role $role)
    {
        if ($role->name === 'Super Admin') {
            return response()->json(['message' => 'The Super Admin role cannot be deleted.'], 403);
        }

        $role->delete();

        Cache::forget('roles');

        return response()->noContent();
    }

    /**
     * Get a count of all roles.
     */
    public function count()
    {
        return Role::count();
    }
}
