<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Booking;
use App\Models\AdminActivityLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ReportController extends Controller
{
    /**
     * Provide a comprehensive report for the admin dashboard.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */public function index(Request $request)
    {
        try {
            // 1. Basic Metrics (OK for MySQL)
            $totalRevenue = Booking::where('status', 'Completed')->sum('price');
            $revenueToday = Booking::where('status', 'Completed')->whereDate('created_at', today())->sum('price');
            $revenueThisWeek = Booking::where('status', 'Completed')->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()])->sum('price');
            $revenueThisMonth = Booking::where('status', 'Completed')->whereYear('created_at', now()->year)->whereMonth('created_at', now()->month)->sum('price');
            $totalBookings = Booking::count();
            $completedBookings = Booking::where('status', 'Completed')->count();
            $pendingBookings = Booking::where('status', 'PENDING')->count();
            $totalUsers = DB::table('users')->count();

            // 2. Average metrics (OK for MySQL)
            $avgBookingValue = $completedBookings > 0 ? $totalRevenue / $completedBookings : 0;
            $bookingsToday = Booking::whereDate('created_at', today())->count();
            $bookingsThisWeek = Booking::whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()])->count();
            $bookingsThisMonth = Booking::whereYear('created_at', now()->year)->whereMonth('created_at', now()->month)->count();

            // 3. Revenue by Game (OK for MySQL)
            $revenueByGame = Booking::where('bookings.status', 'Completed')
                ->join('games', 'bookings.gameName', '=', 'games.name')
                ->select('games.name as game_name', DB::raw('SUM(bookings.price) as total_revenue'), DB::raw('COUNT(bookings.id) as booking_count'))
                ->groupBy('games.name')
                ->orderBy('total_revenue', 'desc')
                ->get();

            // 4. Game Popularity (OK for MySQL)
            $gamePopularity = Booking::join('games', 'bookings.gameName', '=', 'games.name')
                ->select('games.name as game_name', DB::raw('COUNT(bookings.id) as booking_count'), DB::raw('COUNT(CASE WHEN bookings.status = "Completed" THEN 1 END) as completed_count'))
                ->groupBy('games.name')
                ->orderBy('booking_count', 'desc')
                ->get();

            // 5. Daily Revenue - FIXED: Using DATE() in both SELECT and GROUP BY
            $dailyRevenue = Booking::where('status', 'Completed')
                ->select(
                    DB::raw('DATE(created_at) as date_val'),
                    DB::raw('SUM(price) as revenue'),
                    DB::raw('COUNT(id) as bookings')
                )
                ->where('created_at', '>=', now()->subDays(30))
                ->groupBy('date_val')
                ->orderBy('date_val', 'asc')
                ->get()
                ->map(function ($item) {
                    $item->date = Carbon::parse($item->date_val)->format('M d');
                    return $item;
                });

            // 6. Monthly Revenue - FIXED: Already uses DATE_FORMAT (Good for MySQL)
            $monthlyRevenue = Booking::where('status', 'Completed')
                ->select(
                    DB::raw('DATE_FORMAT(created_at, "%Y-%m") as month_val'),
                    DB::raw('SUM(price) as revenue'),
                    DB::raw('COUNT(id) as bookings')
                )
                ->where('created_at', '>=', now()->subMonths(12)->startOfMonth())
                ->groupBy('month_val')
                ->orderBy('month_val', 'asc')
                ->get()
                ->map(function ($item) {
                    $date = Carbon::createFromFormat('Y-m', $item->month_val);
                    $item->month = $date->format('M Y');
                    return $item;
                });

            // 7. Revenue by Hour - FIXED: MySQL HOUR() needs to be in GROUP BY
            $hourlyRevenue = Booking::where('status', 'Completed')
                ->whereDate('created_at', today())
                ->select(
                    DB::raw('HOUR(created_at) as hour_val'),
                    DB::raw('SUM(price) as revenue'),
                    DB::raw('COUNT(id) as bookings')
                )
                ->groupBy('hour_val')
                ->orderBy('hour_val', 'asc')
                ->get()
                ->map(function ($item) {
                    $item->hour = sprintf('%02d:00', $item->hour_val);
                    return $item;
                });

            // 8. Top Performing Games (Already MySQL compatible)
            $topGames = $revenueByGame->take(5)->map(function ($game) use ($gamePopularity) {
                $popularity = $gamePopularity->firstWhere('game_name', $game->game_name);
                return [
                    'name' => $game->game_name,
                    'revenue' => $game->total_revenue,
                    'bookings' => $popularity ? $popularity->booking_count : 0,
                    'avg_value' => $popularity && $popularity->booking_count > 0 ? $game->total_revenue / $popularity->booking_count : 0
                ];
            });

            // Prepare final data array
            $data = [
                'totalRevenue' => (float) $totalRevenue,
                'revenueToday' => (float) $revenueToday,
                'revenueThisWeek' => (float) $revenueThisWeek,
                'revenueThisMonth' => (float) $revenueThisMonth,
                'totalBookings' => $totalBookings,
                'completedBookings' => $completedBookings,
                'pendingBookings' => $pendingBookings,
                'totalUsers' => $totalUsers,
                'avgBookingValue' => round($avgBookingValue, 2),
                'bookingsToday' => $bookingsToday,
                'bookingsThisWeek' => $bookingsThisWeek,
                'bookingsThisMonth' => $bookingsThisMonth,
                'revenueByGame' => $revenueByGame,
                'gamePopularity' => $gamePopularity,
                'dailyRevenue' => $dailyRevenue,
                'monthlyRevenue' => $monthlyRevenue,
                'hourlyRevenue' => $hourlyRevenue,
                'topGames' => $topGames,
            ];

            return response()->json(['success' => true, 'data' => $data]);

        } catch (\Exception $e) {
            \Log::error('Report generation failed: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while generating the report.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Provide a filtered history of bookings.
     *
     * @param  string $query
     * @return \Illuminate\Http\JsonResponse
     */
    public function bookingHistory($query)
    {
        try {
            $filters = json_decode(urldecode($query), true);

            $bookingQuery = Booking::query();

            if (!empty($filters['gameName']) && $filters['gameName'] !== 'all') {
                $bookingQuery->where('gameName', $filters['gameName']);
            }

            if (!empty($filters['startDate']) && !empty($filters['endDate'])) {
                 $bookingQuery->whereBetween('created_at', [
                    Carbon::parse($filters['startDate'])->startOfDay(),
                    Carbon::parse($filters['endDate'])->endOfDay()
                ]);
            }

            $bookings = $bookingQuery->orderBy('created_at', 'desc')->get();

            $formattedBookings = $bookings->map(function($booking) {
                return [
                    'ticketId' => $booking->ticketId,
                    'name' => $booking->name,
                    'email' => $booking->email,
                    'phone' => $booking->phone,
                    'gameName' => $booking->gameName,
                    'price' => $booking->price,
                    'status' => $booking->status,
                    'date' => Carbon::parse($booking->created_at)->toDateString(),
                ];
            });

            return response()->json(['success' => true, 'data' => $formattedBookings]);

        } catch (\Exception $e) {
            \Log::error('Booking history fetch failed: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while fetching booking history.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get Revenue & Financial Reports
     */
    public function revenueReports(Request $request)
    {
        $query = Booking::query();

        // Apply date filters if provided
        if ($request->has('startDate') && $request->startDate !== 'all') {
            $query->whereDate('created_at', '>=', $request->startDate);
        }
        if ($request->has('endDate') && $request->endDate !== 'all') {
            $query->whereDate('created_at', '<=', $request->endDate);
        }

        // Apply status filter if provided
        if ($request->has('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        // Apply game filter if provided
        if ($request->has('gameName') && $request->gameName !== 'all') {
            $query->where('gameName', $request->gameName);
        }

        // Total Gross Revenue
        $totalGrossRevenue = (clone $query)
        ->whereIn('status', ['Admitted', 'Completed'])
        ->sum('price');
        // Revenue by Game
        $revenueByGame = (clone $query)->select('gameName', DB::raw('SUM(price) as total_revenue'))
            ->groupBy('gameName')
            ->orderBy('total_revenue', 'desc')
            ->get();

        // Daily/Monthly Sales Trends
        $dailyTrends = (clone $query)->select(
                DB::raw('DATE(created_at) as date'),
                DB::raw('SUM(price) as revenue'),
                DB::raw('COUNT(*) as bookings')
            )
            ->whereDate('created_at', '>=', Carbon::now()->subDays(30))
            ->groupBy(DB::raw('DATE(created_at)'))
            ->orderBy(DB::raw('DATE(created_at)'), 'asc')
            ->get();

        $monthlyTrends = (clone $query)->select(
                DB::raw('DATE_FORMAT(created_at, "%Y-%m") as month'),
                DB::raw('SUM(price) as revenue'),
                DB::raw('COUNT(*) as bookings')
            )
            ->whereDate('created_at', '>=', Carbon::now()->subMonths(12))
            ->groupBy(DB::raw('DATE_FORMAT(created_at, "%Y-%m")'))
            ->orderBy(DB::raw('DATE_FORMAT(created_at, "%Y-%m")'), 'asc')
            ->get();

        // Average Order Value (AOV)
        $totalBookings = (clone $query)->count();
        $avgOrderValue = $totalBookings > 0 ? $totalGrossRevenue / $totalBookings : 0;

        // Refund/Cancellation Liability
      $refundLiability = (clone $query)->where('status', 'Completed')->sum('price');
        return response()->json([
            'success' => true,
            'data' => [
                'totalGrossRevenue' => (float) $totalGrossRevenue,
                'revenueByGame' => $revenueByGame,
                'dailyTrends' => $dailyTrends,
                'monthlyTrends' => $monthlyTrends,
                'avgOrderValue' => round($avgOrderValue, 2),
                'refundLiability' => (float) $refundLiability,
                'totalBookings' => $totalBookings,
            ]
        ]);
    }

    /**
     * Get Operational & Volume Reports
     */
    public function operationalReports(Request $request)
    {
        $query = Booking::query();

        // Apply date filters if provided
        if ($request->has('startDate') && $request->startDate !== 'all') {
            $query->whereDate('created_at', '>=', $request->startDate);
        }
        if ($request->has('endDate') && $request->endDate !== 'all') {
            $query->whereDate('created_at', '<=', $request->endDate);
        }

        // Apply status filter if provided
        if ($request->has('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        // Apply game filter if provided
        if ($request->has('gameName') && $request->gameName !== 'all') {
            $query->where('gameName', $request->gameName);
        }

        // Ticket Sales Volume
        $ticketSalesVolume = (clone $query)
    ->whereIn('status', ['Completed', 'Admitted'])
    ->sum('quantity');

        // Game Popularity (bookings count vs people count)
      $gamePopularity = (clone $query)->select(
        'gameName',
        DB::raw("COUNT(CASE WHEN status IN ('Completed', 'Admitted') THEN 1 END) as booking_count"),
        DB::raw("SUM(CASE WHEN status IN ('Completed', 'Admitted') THEN quantity ELSE 0 END) as people_count")
    )
    ->groupBy('gameName')
    ->orderBy('booking_count', 'desc')
    ->get();

        // Attendance Manifest (by event date)
 $attendanceManifest = (clone $query)->select(
       
        'gameName',
        // Expected Attendance: Count of ROWS where status is 'Completed'
        DB::raw("COUNT(CASE WHEN status = 'Completed' THEN 1 END) as expected_attendance"),
        // Booking Count: Count of ROWS where status is 'Admitted'
        DB::raw("COUNT(CASE WHEN status = 'Admitted' THEN 1 END) as booking_count")
    )
    ->groupBy('gameName')
    // ->orderBy('date', 'asc')
    ->get();
        // Peak Time Analysis (by day of week)
    $peakTimeAnalysis = (clone $query)->select(
        DB::raw('DAYOFWEEK(created_at) as day_of_week'),
        DB::raw('DAYNAME(created_at) as day_name'),
        // Count only specific statuses
        DB::raw("COUNT(CASE WHEN status IN ('Completed', 'Admitted') THEN 1 END) as booking_count"),
        // Sum quantity only for specific statuses
        DB::raw("SUM(CASE WHEN status IN ('Completed', 'Admitted') THEN quantity ELSE 0 END) as ticket_count")
    )
    ->groupBy(DB::raw('DAYOFWEEK(created_at)'), DB::raw('DAYNAME(created_at)'))
    ->orderBy(DB::raw('DAYOFWEEK(created_at)'), 'asc')
    ->get();
        return response()->json([
            'success' => true,
            'data' => [
                'ticketSalesVolume' => $ticketSalesVolume,
                'gamePopularity' => $gamePopularity,
                'attendanceManifest' => $attendanceManifest,
                'peakTimeAnalysis' => $peakTimeAnalysis,
            ]
        ]);
    }

    /**
     * Get Customer & Marketing Reports
     */
    public function customerReports(Request $request)
    {
        $query = Booking::query();

        // Apply date filters if provided
        if ($request->has('startDate') && $request->startDate !== 'all') {
            $query->whereDate('created_at', '>=', $request->startDate);
        }
        if ($request->has('endDate') && $request->endDate !== 'all') {
            $query->whereDate('created_at', '<=', $request->endDate);
        }

        // Apply status filter if provided
        if ($request->has('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        // Apply game filter if provided
        if ($request->has('gameName') && $request->gameName !== 'all') {
            $query->where('gameName', $request->gameName);
        }

        // Customer Directory
        $customerDirectory = (clone $query)->select('name', 'email', 'phone', 'created_at')
            ->orderBy('created_at', 'desc')
            ->limit(1000)
            ->get();

        // Bulk Booking Report (high-value customers)
$bulkBookings = (clone $query)->select([
        'name', 
        'email', 
        'phone', 
        'gameName', 
        'quantity', 
        'price',
        'price as total_value', // Laravel handles "as" aliases in select automatically
        'created_at'
    ])
    ->orderBy('price', 'desc')
    ->get();

        // Repeat Customer Detection
        $repeatCustomers = DB::table('bookings')
            ->select('email', DB::raw('COUNT(*) as booking_count'), DB::raw('SUM(price) as total_spent'))
            ->whereNotNull('email')
            ->where('email', '!=', '')
            ->groupBy('email')
            ->having('booking_count', '>', 1)
            ->orderBy('booking_count', 'desc')
            ->limit(100)
            ->get();

        // Geographic Lead Generation (area codes)
        $geographicData = (clone $query)->select(
                DB::raw('SUBSTRING(phone, 1, 4) as area_code'),
                DB::raw('COUNT(*) as customer_count'),
                DB::raw('SUM(price) as total_revenue')
            )
            ->whereNotNull('phone')
            ->where('phone', '!=', '')
            ->groupBy(DB::raw('SUBSTRING(phone, 1, 4)'))
            ->orderBy('customer_count', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'customerDirectory' => $customerDirectory,
                'bulkBookings' => $bulkBookings,
                'repeatCustomers' => $repeatCustomers,
                'geographicData' => $geographicData,
            ]
        ]);
    }

    /**
     * Get Logistics & Technical Reports
     */
    public function logisticsReports(Request $request)
    {
        $query = Booking::query();

        // Apply date filters if provided
        if ($request->has('startDate') && $request->startDate !== 'all') {
            $query->whereDate('created_at', '>=', $request->startDate);
        }
        if ($request->has('endDate') && $request->endDate !== 'all') {
            $query->whereDate('created_at', '<=', $request->endDate);
        }

        // Ticket Status Audit
        $statusAudit = (clone $query)->select(
                'status',
                DB::raw('COUNT(*) as count'),
                DB::raw('SUM(price) as total_value')
            )
            ->groupBy('status')
            ->orderBy('count', 'desc')
            ->get();

        // Payment Status Summary
        $paymentStatusSummary = (clone $query)->select(
                'payment_status_description',
                DB::raw('COUNT(*) as count'),
                DB::raw('SUM(price) as total_value')
            )
            ->whereNotNull('payment_status_description')
            ->groupBy('payment_status_description')
            ->orderBy('count', 'desc')
            ->get();

        // Queue Management Summary
        $queueSummary = (clone $query)->select(
                'gameName',
                DB::raw('COUNT(CASE WHEN queuePosition IS NOT NULL THEN 1 END) as queued_bookings'),
                DB::raw('AVG(queuePosition) as avg_queue_position'),
                DB::raw('MAX(queuePosition) as max_queue_position')
            )
            ->groupBy('gameName')
            ->orderBy('queued_bookings', 'desc')
            ->get();

        // System Health Metrics
        $systemHealth = [
            'total_bookings' => (clone $query)->count(),
            'completed_bookings' => (clone $query)->whereIn('status', ['Admitted', 'Completed'])->count(),
            'pending_bookings' => (clone $query)->where('status', 'Pending')->count(),
            'cancelled_bookings' => (clone $query)->where('status', 'Cancelled')->count(),
            'avg_processing_time' => (clone $query)->whereNotNull('updated_at')
                ->selectRaw('AVG(TIMESTAMPDIFF(MINUTE, created_at, updated_at)) as avg_minutes')
                ->first()->avg_minutes ?? 0,
        ];

        return response()->json([
            'success' => true,
            'data' => [
                'statusAudit' => $statusAudit,
                'paymentStatusSummary' => $paymentStatusSummary,
                'queueSummary' => $queueSummary,
                'systemHealth' => $systemHealth,
            ]
        ]);
    }

    /**
     * Export bookings data to CSV format.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Symfony\Component\HttpFoundation\StreamedResponse
     */
    public function exportCsv(Request $request)
    {
        try {
            // Build the base query with filters
            // $query = Booking::with(['user', 'game']);
              $query = Booking::query();

            // Apply date filter
            if ($request->has('startDate') && $request->has('endDate')) {
                $query->whereBetween('bookings.created_at', [
                    Carbon::parse($request->startDate)->startOfDay(),
                    Carbon::parse($request->endDate)->endOfDay()
                ]);
            }

            // Apply status filter
            if ($request->has('status') && $request->status !== 'all') {
                $query->where('bookings.status', $request->status);
            }

            // Apply game filter
            if ($request->has('gameName') && $request->gameName !== 'all') {
                $query->where('bookings.gameName', $request->gameName);
            }

            $bookings = $query->orderBy('created_at', 'desc')->get();

            $filename = 'bookings_report_' . now()->format('Y-m-d_H-i-s') . '.csv';

            return response()->stream(function () use ($bookings) {
                $handle = fopen('php://output', 'w');

                // Write CSV headers
                fputcsv($handle, [
                    'Booking ID',
                    'Customer Name',
                    'Customer Email',
                    'Game Name',
                    'Quantity',
                    'Price',
                    'Total Amount',
                    'Status',
                    'Payment Status',
                    'Booking Date',
                    'Updated Date'
                ]);

                // Write booking data
                foreach ($bookings as $booking) {
                    fputcsv($handle, [
                        $booking->id,
                        $booking->user->name ?? 'N/A',
                        $booking->user->email ?? 'N/A',
                        $booking->gameName,
                        $booking->quantity,
                        number_format($booking->price, 2),
                        number_format($booking->price , 2),
                        $booking->status,
                        $booking->payment_status_description ?? 'N/A',
                        $booking->created_at->format('Y-m-d H:i:s'),
                        $booking->updated_at->format('Y-m-d H:i:s')
                    ]);
                }

                fclose($handle);
            }, 200, [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => 'attachment; filename="' . $filename . '"',
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to export CSV: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Export financial report to PDF format.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function exportPdf(Request $request)
    {
        try {
            // Build the base query with filters
            // $query = Booking::with(['user', 'game']);
            $query = Booking::query();

            // Apply date filter
            if ($request->has('startDate') && $request->has('endDate')) {
                $query->whereBetween('bookings.created_at', [
                    Carbon::parse($request->startDate)->startOfDay(),
                    Carbon::parse($request->endDate)->endOfDay()
                ]);
            }

            // Apply status filter
            if ($request->has('status') && $request->status !== 'all') {
                $query->where('bookings.status', $request->status);
            }

            // Apply game filter
            if ($request->has('gameName') && $request->gameName !== 'all') {
                $query->where('bookings.gameName', $request->gameName);
            }

            // Get summary data for the report
            $bookings = $query->orderBy('created_at', 'desc')->get();
            $totalRevenue = $bookings->where('status', 'Completed')->sum(function ($booking) {
                return $booking->price ;
            });
            $totalBookings = $bookings->count();
            $completedBookings = $bookings->where('status', 'Completed')->count();

            // Revenue by game
            $revenueByGame = $bookings->where('status', 'Completed')
                ->groupBy('gameName')
                ->map(function ($gameBookings, $gameName) {
                    return [
                        'game' => $gameName,
                        'revenue' => $gameBookings->sum(function ($booking) {
                            return $booking->price ;
                        }),
                        'bookings' => $gameBookings->count()
                    ];
                })->sortByDesc('revenue');

            $html = $this->generatePdfHtml($bookings, $totalRevenue, $totalBookings, $completedBookings, $revenueByGame, $request);

            $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadHTML($html);
            $pdf->setPaper('a4', 'landscape');

            $filename = 'financial_report_' . now()->format('Y-m-d_H-i-s') . '.pdf';

            return $pdf->download($filename);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to export PDF: ' . $e->getMessage()
            ], 500);
        }
    }
/**
     * Generate HTML content for PDF report with a dark, professional theme.
     */
    private function generatePdfHtml($bookings, $totalRevenue, $totalBookings, $completedBookings, $revenueByGame, $request)
    {
        $dateRange = 'All Time';
        if ($request->filled('startDate') && $request->filled('endDate')) {
            $dateRange = Carbon::parse($request->startDate)->format('M d, Y') . ' - ' . Carbon::parse($request->endDate)->format('M d, Y');
        }

        // Calculations
        $pendingBookings = $bookings->where('status', 'Pending')->count();
        $cancelledBookings = $bookings->whereIn('status', ['Cancelled', 'INVALID', 'FAILED'])->count();
        $avgBookingValue = $completedBookings > 0 ? $totalRevenue / $completedBookings : 0;

        $html = '
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="utf-8">
            <style>
                @page { margin: 0.5in; background-color: #000000; }
                body {
                    font-family: "Helvetica Neue", Arial, sans-serif;
                    background-color: #000000;
                    color: #ffffff;
                    font-size: 14px; /* Increased font size */
                    margin: 0;
                    padding: 0;
                }
                .container { padding: 20px; }
                
                /* Layout Tables */
                .w-full { width: 100%; border-collapse: collapse; }
                
                /* Heading Section */
                .header-table { margin-bottom: 40px; border-bottom: 2px solid #2563eb; }
                .header-title { font-size: 32px; font-weight: bold; color: #3b82f6; }
                .header-meta { text-align: right; color: #94a3b8; font-size: 12px; }

                /* KPI Table Styles */
                .kpi-table { margin-bottom: 40px; }
                .kpi-box { 
                    background-color: #111827; 
                    border: 1px solid #1e293b; 
                    padding: 25px; 
                    text-align: center;
                }
                .kpi-label { color: #94a3b8; font-size: 12px; text-transform: uppercase; letter-spacing: 1px; margin-bottom: 10px; display: block; }
                .kpi-value { font-size: 26px; font-weight: bold; color: #ffffff; }

                /* Section Headings */
                .section-title { 
                    background-color: #1e293b; 
                    padding: 12px 20px; 
                    font-size: 18px; 
                    font-weight: bold; 
                    color: #3b82f6; 
                    margin-bottom: 20px;
                    border-left: 5px solid #2563eb;
                }

                /* Data Tables */
                .data-table { margin-bottom: 40px; }
                .data-table th { 
                    background-color: #1e3a8a; 
                    color: #ffffff; 
                    text-align: left; 
                    padding: 15px; 
                    font-size: 12px; 
                    text-transform: uppercase; 
                }
                .data-table td { 
                    padding: 15px; 
                    border-bottom: 1px solid #1e293b; 
                    color: #e2e8f0;
                }
                .data-table tr:nth-child(even) { background-color: #0a0a0a; }
                
                /* Status Colors */
                .status-badge { padding: 4px 10px; border-radius: 4px; font-size: 11px; font-weight: bold; }
                .status-success { background-color: #064e3b; color: #6ee7b7; }
                .status-warn { background-color: #78350f; color: #fcd34d; }
                .status-danger { background-color: #7f1d1d; color: #fca5a5; }

                .text-blue { color: #3b82f6; }
                .text-green { color: #10b981; }
            </style>
        </head>
        <body>
            <div class="container">
                <table class="w-full header-table">
                    <tr>
                        <td class="header-title" style="padding-bottom: 10px;">FINANCIAL REPORT</td>
                        <td class="header-meta">
                            <strong>PERIOD:</strong> '.$dateRange.'<br>
                            <strong>GENERATED:</strong> '.now()->format('M d, Y H:i').'
                        </td>
                    </tr>
                </table>

                <table class="w-full kpi-table">
                    <tr>
                        <td width="33%">
                            <div class="kpi-box">
                                <span class="kpi-label">Gross Revenue</span>
                                <span class="kpi-value text-green">$'.number_format($totalRevenue, 2).'</span>
                            </div>
                        </td>
                        <td width="33%" style="padding: 0 15px;">
                            <div class="kpi-box">
                                <span class="kpi-label">Total Bookings</span>
                                <span class="kpi-value">'.number_format($totalBookings).'</span>
                            </div>
                        </td>
                        <td width="33%">
                            <div class="kpi-box">
                                <span class="kpi-label">Avg Order Value</span>
                                <span class="kpi-value text-blue">$'.number_format($avgBookingValue, 2).'</span>
                            </div>
                        </td>
                    </tr>
                </table>

                <div class="section-title">REVENUE ANALYSIS BY GAME</div>
                <table class="w-full data-table">
                    <thead>
                        <tr>
                            <th>Game Name</th>
                            <th>Total Revenue</th>
                            <th>Bookings Count</th>
                            <th>Avg. per Booking</th>
                            <th>Contribution %</th>
                        </tr>
                    </thead>
                    <tbody>';
        foreach ($revenueByGame as $game) {
            $avgPer = $game['bookings'] > 0 ? $game['revenue'] / $game['bookings'] : 0;
            $contribution = $totalRevenue > 0 ? ($game['revenue'] / $totalRevenue) * 100 : 0;
            $html .= '
                        <tr>
                            <td><strong>'.$game['game'].'</strong></td>
                            <td class="text-green">$'.number_format($game['revenue'], 2).'</td>
                            <td>'.$game['bookings'].'</td>
                            <td>$'.number_format($avgPer, 2).'</td>
                            <td class="text-blue">'.number_format($contribution, 1).'%</td>
                        </tr>';
        }
        $html .= '
                    </tbody>
                </table>

                <div class="section-title">BOOKING STATUS SUMMARY</div>
                <table class="w-full data-table">
                    <thead>
                        <tr>
                            <th>Status Metric</th>
                            <th>Total Count</th>
                            <th>Percentage of Total</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td>Successful / Admitted</td>
                            <td class="text-green">'.number_format($completedBookings).'</td>
                            <td>'.($totalBookings > 0 ? round(($completedBookings / $totalBookings) * 100, 1) : 0).'%</td>
                        </tr>
                        <tr>
                            <td>Pending Processing</td>
                            <td class="text-warn">'.number_format($pendingBookings).'</td>
                            <td>'.($totalBookings > 0 ? round(($pendingBookings / $totalBookings) * 100, 1) : 0).'%</td>
                        </tr>
                        <tr>
                            <td>Cancelled / Failed</td>
                            <td class="text-danger">'.number_format($cancelledBookings).'</td>
                            <td>'.($totalBookings > 0 ? round(($cancelledBookings / $totalBookings) * 100, 1) : 0).'%</td>
                        </tr>
                    </tbody>
                </table>

                <div class="section-title">RECENT TRANSACTION LOG</div>
                <table class="w-full data-table">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Customer Name</th>
                            <th>Game</th>
                            <th>Total Amount</th>
                            <th>Status</th>
                            <th>Date</th>
                        </tr>
                    </thead>
                    <tbody>';
        foreach ($bookings->take(15) as $booking) {
            $statusClass = 'status-warn';
            if (in_array($booking->status, ['Admitted', 'Completed', 'Paid'])) $statusClass = 'status-success';
            if (in_array($booking->status, ['Cancelled', 'FAILED', 'INVALID'])) $statusClass = 'status-danger';

            $html .= '
                        <tr>
                            <td>#'.$booking->id.'</td>
                            <td>'.($booking->name ?? 'Guest User').'</td>
                            <td>'.$booking->gameName.'</td>
                            <td class="text-green">$'.number_format($booking->price , 2).'</td>
                            <td><span class="status-badge '.$statusClass.'">'.$booking->status.'</span></td>
                            <td>'.$booking->created_at->format('M d, Y').'</td>
                        </tr>';
        }
        $html .= '
                    </tbody>
                </table>

                <table class="w-full" style="margin-top: 50px; border-top: 1px solid #1e293b;">
                    <tr>
                        <td style="padding-top: 20px; text-align: center; color: #64748b; font-size: 10px;">
                            CONFIDENTIAL FINANCIAL DOCUMENT • ARCADEA BOOKING SYSTEM • Page 1 of 1
                        </td>
                    </tr>
                </table>
            </div>
        </body>
        </html>';

        return $html;
    }

    /**
     * Get Admin Performance & Activity Reports
     */
    public function adminPerformanceReports(Request $request)
    {
        $query = AdminActivityLog::query();

        // Apply date filters if provided
        if ($request->has('startDate') && $request->startDate !== 'all') {
            $query->whereDate('created_at', '>=', $request->startDate);
        }
        if ($request->has('endDate') && $request->endDate !== 'all') {
            $query->whereDate('created_at', '<=', $request->endDate);
        }

        // Apply admin filter if provided
        if ($request->has('adminEmail') && $request->adminEmail !== 'all') {
            $query->whereHas('user', function ($q) use ($request) {
                $q->where('email', $request->adminEmail);
            });
        }

        // Total activities
        $totalActivities = (clone $query)->count();

        // Activities by admin
        $activitiesByAdmin = (clone $query)->select(
                'admin_activity_logs.user_id',
                'users.name',
                'users.email',
                DB::raw('COUNT(*) as total_activities'),
                DB::raw('COUNT(CASE WHEN action = "admitted_ticket" THEN 1 END) as tickets_admitted'),
                DB::raw('COUNT(CASE WHEN action = "updated_booking" THEN 1 END) as bookings_updated'),
                DB::raw('COUNT(CASE WHEN action = "checked_payment_status" THEN 1 END) as status_checks'),
                DB::raw('COUNT(CASE WHEN action = "refunded_payment" THEN 1 END) as refunds_processed')
            )
            ->leftJoin('users', 'admin_activity_logs.user_id', '=', 'users.id')
            ->groupBy('admin_activity_logs.user_id', 'users.name', 'users.email')
            ->orderBy('total_activities', 'desc')
            ->get()
            ->map(function ($item) {
                return [
                    'user_id' => $item->user_id,
                    'admin' => [
                        'name' => $item->name ?? 'Deleted User',
                        'email' => $item->email ?? '',
                    ],
                    'total_activities' => $item->total_activities,
                    'tickets_admitted' => $item->tickets_admitted,
                    'bookings_updated' => $item->bookings_updated,
                    'status_checks' => $item->status_checks,
                    'refunds_processed' => $item->refunds_processed,
                ];
            });

        // Activities by action type
        $activitiesByAction = (clone $query)->select(
                'action',
                DB::raw('COUNT(*) as count')
            )
            ->groupBy('action')
            ->orderBy('count', 'desc')
            ->get();

        // Daily activity trends
        $dailyTrends = (clone $query)->select(
                DB::raw('DATE(created_at) as date'),
                DB::raw('COUNT(*) as activities')
            )
            ->whereDate('created_at', '>=', Carbon::now()->subDays(30))
            ->groupBy(DB::raw('DATE(created_at)'))
            ->orderBy(DB::raw('DATE(created_at)'), 'asc')
            ->get();

        // Top active admins
        $topAdmins = $activitiesByAdmin->take(10);

        // Recent Activity Logs (Detailed)
        $recentActivities = (clone $query)->select(
                'admin_activity_logs.id',
                'admin_activity_logs.action',
                'admin_activity_logs.created_at',
                'users.name as user_name',
                'users.email as user_email'
            )
            ->leftJoin('users', 'admin_activity_logs.user_id', '=', 'users.id')
            ->orderBy('admin_activity_logs.created_at', 'desc')
            ->limit(100)
            ->get()
            ->map(function ($item) {
                return [
                    'id' => $item->id,
                    'action' => $item->action,
                    'user_name' => $item->user_name ?? 'System/Deleted',
                    'user_email' => $item->user_email ?? 'N/A',
                    'transaction_date' => $item->created_at->toDateTimeString(),
                ];
            });

        return response()->json([
            'success' => true,
            'data' => [
                'totalActivities' => $totalActivities,
                'activitiesByAdmin' => $activitiesByAdmin,
                'activitiesByAction' => $activitiesByAction,
                'dailyTrends' => $dailyTrends,
                'topAdmins' => $topAdmins,
                'recentActivities' => $recentActivities,
            ]
        ]);
    }
}
