<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use App\Models\Booking;
use App\Models\AdminActivityLog;
use Exception;
use Illuminate\Validation\ValidationException;
use Carbon\Carbon;

class PesapalController extends Controller
{
    private function getAuthToken(): array
    {
        if (Cache::has('pesapal_token')) {
            return [Cache::get('pesapal_token'), ['[Auth] Using cached token.']];
        }
        $logs = ['[Auth] No valid token in cache, requesting a new one...'];
        $response = Http::post(config('pesapal.api_url') . '/api/Auth/RequestToken', [
            'consumer_key' => config('pesapal.consumer_key'),
            'consumer_secret' => config('pesapal.consumer_secret'),
        ]);
        $data = $response->json();
        if ($response->successful() && isset($data['token'])) {
            Cache::put('pesapal_token', $data['token'], now()->parse($data['expiryDate'])->subMinutes(5));
            return [$data['token'], $logs];
        }
        throw new Exception($data['error']['message'] ?? 'Failed to obtain Pesapal auth token.');
    }

    public function submitOrder(Request $request)
    {
        $logs = ['[Client] Initiating payment process...'];
        try {
            $validated = $request->validate([
                'name' => 'required|string|min:2',
                'email' => 'required|email',
                'phone' => 'nullable|string',
                'gameName' => 'required|string',
                'price' => 'required|numeric|min:1',
                'quantity' => 'required|integer|min:1',
                'ticketId' => 'required|string|unique:bookings,ticketId'
            ]);
            $booking = Booking::create([
                'ticketId' => $validated['ticketId'],
                'name' => $validated['name'],
                'email' => $validated['email'],
                'phone' => $validated['phone'],
                'gameName' => $validated['gameName'],
                'price' => $validated['price'],
                'quantity' => $validated['quantity'],
                'date' => now(),
                'status' => 'Pending',
            ]);
            $logs[] = '[Server] Booking created with PENDING status.';

            list($token, $authLogs) = $this->getAuthToken();
            $logs = array_merge($logs, $authLogs);

            $nameParts = explode(' ', $validated['name']);
            $firstName = array_shift($nameParts);
            $lastName = implode(' ', $nameParts);

            $orderPayload = [
                'id' => $validated['ticketId'],
                'currency' => 'KES',
                'amount' => $validated['price'],
                'description' => 'Payment for ' . $validated['gameName'],
                'callback_url' => config('app.frontend_url') . '/confirmation',
                'notification_id' => config('pesapal.ipn_id'),
                'billing_address' => ['email_address' => $validated['email'], 'phone_number' => $validated['phone'] ?? '', 'country_code' => 'KE', 'first_name' => $firstName, 'last_name' => $lastName ?: $firstName]
            ];
            
            $response = Http::withHeaders(['Authorization' => 'Bearer ' . $token, 'Accept' => 'application/json', 'Content-Type' => 'application/json'])->post(config('pesapal.api_url') . '/api/Transactions/SubmitOrderRequest', $orderPayload);
            $responseData = $response->json();

            if ($response->successful() && isset($responseData['order_tracking_id'])) {
                $booking->update(['order_tracking_id' => $responseData['order_tracking_id']]);
                return response()->json(['success' => true, 'message' => 'Order created. Redirecting...', 'redirect_url' => $responseData['redirect_url'], 'logs' => $logs]);
            } else {
                $booking->update(['status' => 'Failed']);
                throw new Exception($responseData['error']['message'] ?? 'Failed to get payment URL.');
            }
        } catch (ValidationException $e) {
            return response()->json(['success' => false, 'message' => 'Invalid data.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            Log::error('[PesapalController] Error in submitOrder.', ['exception' => $e->getMessage()]);
            return response()->json(['success' => false, 'message' => $e->getMessage(), 'logs' => $logs], 500);
        }
    }
    
    public function getTransactionStatus(Request $request)
    {
        $orderTrackingId = $request->query('orderTrackingId');
        $logs = ["[Server] Checking status for OrderTrackingId: {$orderTrackingId}"];

        if (!$orderTrackingId) {
            return response()->json(['success' => false, 'message' => 'orderTrackingId is required.', 'logs' => $logs], 422);
        }

        try {
            list($token, $authLogs) = $this->getAuthToken();
            $logs = array_merge($logs, $authLogs);

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
                'Accept' => 'application/json'
            ])->get(config('pesapal.api_url') . "/api/Transactions/GetTransactionStatus?orderTrackingId={$orderTrackingId}");

            $responseData = $response->json();
            $logs[] = '[Server] Response from Pesapal: ' . json_encode($responseData);

            if (isset($responseData['error']) && !empty($responseData['error']['code'])) {
                $logs[] = "[Server] Pesapal returned an error: " . ($responseData['error']['message'] ?? 'Unknown error');
                return response()->json([
                    'success' => false,
                    'message' => 'Pesapal returned an error.',
                    'status' => $responseData,
                    'logs' => $logs
                ], 200);
            }

            $booking = null;
            $merchantReference = $responseData['merchant_reference'] ?? null;

            if ($merchantReference) {
                $trimmedReference = trim($merchantReference);
                $logs[] = "[Server] Searching for booking with trimmed, case-insensitive merchant_reference: '{$trimmedReference}'.";
                $booking = Booking::whereRaw('LOWER(ticketId) = ?', [strtolower($trimmedReference)])->first();
            } else {
                $logs[] = "[Server] No merchant_reference found in Pesapal response.";
            }
            
            if ($booking) {
                $newStatus = $responseData['payment_status_description'] ?? 'Failed';
                if ($newStatus === 'COMPLETED') $newStatus = 'Completed';
                if ($newStatus === 'PENDING') $newStatus = 'Pending';
                if ($newStatus === 'FAILED' || $newStatus === 'INVALID') $newStatus = 'Failed';

                $booking->update([
                    'payment_method' => $responseData['payment_method'] ?? null,
                    'amount' => $responseData['amount'] ?? null,
                    'pesapal_created_date' => isset($responseData['created_date']) ? Carbon::parse($responseData['created_date'])->toDateTimeString() : now(),
                    'confirmation_code' => $responseData['confirmation_code'] ?? null,
                    'payment_status_description' => $responseData['payment_status_description'] ?? null,
                    'payment_account' => $responseData['payment_account'] ?? null,
                    'call_back_url' => $responseData['call_back_url'] ?? null,
                    'status_code' => $responseData['status_code'] ?? null,
                    'currency' => $responseData['currency'] ?? null,
                    'status' => $newStatus,
                ]);
                $logs[] = "[Server] Booking (ID: {$booking->id}) status updated to '{$newStatus}'.";
            } else if ($merchantReference) {
                $logs[] = "[Server] CRITICAL: Booking not found for merchant_reference '{$merchantReference}'. The database query returned no results.";
            }

            return response()->json([
                'success' => true,
                'message' => 'Status retrieved successfully.',
                'status' => $responseData,
                'booking' => $booking,
                'logs' => $logs
            ], 200);

        } catch (Exception $e) {
            $detailedError = $e->getMessage() . ' on line ' . $e->getLine() . ' in ' . $e->getFile();
            Log::error('[PesapalController] An unexpected error occurred in getTransactionStatus.', ['exception' => $detailedError]);
            $logs[] = '[Server] CRITICAL: An unexpected error occurred. ' . $detailedError;

            return response()->json([
                'success' => false,
                'message' => 'An unexpected server error occurred.',
                'logs' => $logs
            ], 200);
        }
    }

    public function checkStatusByTrackingId($orderTrackingId)
    {
        $logs = ["[Server] Checking status for OrderTrackingId: {$orderTrackingId}"];
        try {
            $booking = Booking::where('order_tracking_id', $orderTrackingId)->firstOrFail();

            list($token, $authLogs) = $this->getAuthToken();
            $logs = array_merge($logs, $authLogs);

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
                'Accept' => 'application/json'
            ])->get(config('pesapal.api_url') . "/api/Transactions/GetTransactionStatus?orderTrackingId={$orderTrackingId}");

            $responseData = $response->json();
            $logs[] = '[Server] Response from Pesapal: ' . json_encode($responseData);

            // if (isset($responseData['error']) && !empty($responseData['error']['code'])) {
            //     throw new Exception($responseData['error']['message'] ?? 'Pesapal returned an error during status check.');
            // }

            $newStatus = $responseData['payment_status_description'] ?? 'Failed';
            if ($newStatus === 'COMPLETED') $newStatus = 'Completed';
            if ($newStatus === 'PENDING') $newStatus = 'Pending';
            if ($newStatus === 'FAILED' || $newStatus === 'INVALID') $newStatus = 'Failed';

            $booking->update([
                'payment_method' => $responseData['payment_method'] ?? $booking->payment_method,
                'amount' => $responseData['amount'] ?? $booking->amount,
                'pesapal_created_date' => isset($responseData['created_date']) ? Carbon::parse($responseData['created_date'])->toDateTimeString() : $booking->pesapal_created_date,
                'confirmation_code' => $responseData['confirmation_code'] ?? $booking->confirmation_code,
                'payment_status_description' => $responseData['payment_status_description'] ?? $booking->payment_status_description,
                'payment_account' => $responseData['payment_account'] ?? $booking->payment_account,
                'status_code' => $responseData['status_code'] ?? $booking->status_code,
                'currency' => $responseData['currency'] ?? $booking->currency,
                'status' => $newStatus,
            ]);

            $logs[] = "[Server] Booking (ID: {$booking->id}) status updated to '{$newStatus}'.";

            // Log admin activity
            AdminActivityLog::create([
                'user_id' => auth()->id(),
                'action' => 'checked_payment_status',
                'details' => [
                    'booking_id' => $booking->id,
                    'order_tracking_id' => $orderTrackingId,
                    'new_status' => $newStatus,
                ],
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);

            return response()->json(['success' => true, 'message' => 'Booking status updated successfully.', 'booking' => $booking->fresh(), 'logs' => $logs]);

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json(['success' => false, 'message' => 'Booking not found with the given Order Tracking ID.'], 404);
        } catch (Exception $e) {
            $detailedError = $e->getMessage() . ' on line ' . $e->getLine() . ' in ' . $e->getFile();
            Log::error('[PesapalController] An unexpected error occurred in checkStatusByTrackingId.', ['exception' => $detailedError]);
            $logs[] = '[Server] CRITICAL: An unexpected error occurred. ' . $detailedError;

            return response()->json(['success' => false, 'message' => 'An unexpected server error occurred.', 'logs' => $logs], 500);
        }
    }

    /**
 * Register a new IPN URL with Pesapal.
 * This returns a notification_id (GUID) which you must save in your .env or DB.
 */
public function registerIpn(Request $request)
{
    $logs = ['[IPN] Initiating IPN registration...'];
    try {
        $validated = $request->validate([
            'url' => 'required|url', // The public URL Pesapal will send notifications to
            'ipn_notification_type' => 'required|in:GET,POST',
        ]);

        list($token, $authLogs) = $this->getAuthToken();
        $logs = array_merge($logs, $authLogs);

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $token,
            'Accept' => 'application/json',
            'Content-Type' => 'application/json'
        ])->post(config('pesapal.api_url') . '/api/URLSetup/RegisterIPN', [
            'url' => $validated['url'],
            'ipn_notification_type' => $validated['ipn_notification_type']
        ]);

        $responseData = $response->json();

        if ($response->successful()) {
            return response()->json([
                'success' => true, 
                'message' => 'IPN Registered successfully.', 
                'data' => $responseData, 
                'logs' => $logs
            ]);
        }

        throw new Exception($responseData['error']['message'] ?? 'Failed to register IPN.');
    } catch (Exception $e) {
        return response()->json(['success' => false, 'message' => $e->getMessage(), 'logs' => $logs], 500);
    }
}

/**
 * List all registered IPNs for your merchant account.
 */
public function listIpns()
{
    $logs = ['[IPN] Fetching registered IPNs...'];
    try {
        list($token, $authLogs) = $this->getAuthToken();
        $logs = array_merge($logs, $authLogs);

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $token,
            'Accept' => 'application/json'
        ])->get(config('pesapal.api_url') . '/api/URLSetup/GetIpnList');

        if ($response->successful()) {
            return response()->json([
                'success' => true, 
                'ipns' => $response->json(), 
                'logs' => $logs
            ]);
        }

        return response()->json(['success' => false, 'message' => 'Could not fetch IPN list.'], 500);
    } catch (Exception $e) {
        return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
    }
}

/**
 * Process a refund for a completed payment.
 */
public function refundPayment(Request $request)
{
    $logs = ['[Refund] Initiating refund process...'];
    try {
        $validated = $request->validate([
            'confirmation_code' => 'required|string',
            'amount' => 'required|numeric|min:0.01',
            'username' => 'required|string',
            'remarks' => 'required|string|max:255',
        ]);

        // Find the booking by confirmation_code
        $booking = Booking::where('confirmation_code', $validated['confirmation_code'])->first();
        if (!$booking) {
            return response()->json(['success' => false, 'message' => 'Booking not found with the given confirmation code.'], 404);
        }

        // Check if status is Completed
        if ($booking->status !== 'Completed') {
            return response()->json(['success' => false, 'message' => 'Refund can only be processed for completed payments.'], 400);
        }

        // Check amount doesn't exceed original
        if ($validated['amount'] > $booking->price) {
            return response()->json(['success' => false, 'message' => 'Refund amount cannot exceed the original payment amount.'], 400);
        }

        list($token, $authLogs) = $this->getAuthToken();
        $logs = array_merge($logs, $authLogs);

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $token,
            'Accept' => 'application/json',
            'Content-Type' => 'application/json'
        ])->post(config('pesapal.api_url') . '/api/Transactions/RefundRequest', [
            'confirmation_code' => $validated['confirmation_code'],
            'amount' => $validated['amount'],
            'username' => $validated['username'],
            'remarks' => $validated['remarks'],
        ]);

        $responseData = $response->json();
        $logs[] = '[Refund] Response from Pesapal: ' . json_encode($responseData);

        if ($response->successful() && isset($responseData['status']) && $responseData['status'] == 200) {
            // Update booking status to Refunded or partial
            if ($validated['amount'] == $booking->price) {
                $booking->update(['status' => 'Refunded']);
            } else {
                $booking->update(['status' => 'Partially Refunded']);
            }

            // Log admin activity
            AdminActivityLog::create([
                'user_id' => auth()->id(),
                'action' => 'refunded_payment',
                'details' => [
                    'booking_id' => $booking->id,
                    'confirmation_code' => $validated['confirmation_code'],
                    'amount' => $validated['amount'],
                    'remarks' => $validated['remarks'],
                ],
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Refund request submitted successfully.',
                'data' => $responseData,
                'logs' => $logs
            ]);
        } else {
            throw new Exception($responseData['message'] ?? 'Failed to process refund.');
        }
    } catch (ValidationException $e) {
        return response()->json(['success' => false, 'message' => 'Invalid data.', 'errors' => $e->errors()], 422);
    } catch (Exception $e) {
        Log::error('[PesapalController] Error in refundPayment.', ['exception' => $e->getMessage()]);
        return response()->json(['success' => false, 'message' => $e->getMessage(), 'logs' => $logs], 500);
    }
}
}