<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Booking;
use App\Models\AdminActivityLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class BookingController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        Log::channel('stderr')->info('BookingController::index called');
        $query = Booking::query();

        if ($request->has('gameName') && $request->gameName !== 'all') {
            $query->where('gameName', $request->gameName);
        }

        if ($request->has('startDate') && $request->startDate !== 'all') {
            $query->whereDate('created_at', '>=', $request->startDate);
        }

        return $query->get();
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        Log::channel('stderr')->info('BookingController::store called');
        $booking = Booking::create($request->all());
        return response()->json($booking, 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(Booking $booking)
    {
        Log::channel('stderr')->info('BookingController::show called');
        return $booking;
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Booking $booking)
    {
        Log::channel('stderr')->info('BookingController::update called');
        $original = $booking->getOriginal();
        $booking->update($request->all());

        // Log admin activity
        AdminActivityLog::create([
            'user_id' => auth()->id(),
            'action' => 'updated_booking',
            'details' => [
                'booking_id' => $booking->id,
                'changes' => array_diff_assoc($booking->getAttributes(), $original),
            ],
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
        ]);

        return response()->json($booking, 200);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Booking $booking)
    {
        Log::channel('stderr')->info('BookingController::destroy called');
        $booking->delete();
        return response()->json(null, 204);
    }

    /**
     * Get booking history for a user by email, phone, or ticketId.
     */
    public function getHistory($query)
    {
        Log::channel('stderr')->info('BookingController::getHistory called');
        $lowerQuery = strtolower($query);
        $bookings = Booking::whereRaw('LOWER(email) = ?', [$lowerQuery])
                            ->orWhereRaw('LOWER(phone) = ?', [$lowerQuery])
                            ->orWhereRaw('LOWER(ticketId) = ?', [$lowerQuery])
                            ->orderBy('created_at', 'desc')
                            ->get();
        return response()->json($bookings);
    }

    /**
     * Get booking history with optional filters.
     */
    public function history(Request $request)
    {
        Log::channel('stderr')->info('BookingController::history called');
        $query = Booking::query();

        if ($request->has('gameName') && $request->gameName !== 'all') {
            $query->where('gameName', $request->gameName);
        }

        if ($request->has('startDate') && $request->startDate !== 'all') {
            $query->whereDate('created_at', '>=', $request->startDate);
        }

        $bookings = $query->get()->map(function ($booking) {
            return [
                'ticketId' => $booking->ticketId,
                'name' => $booking->name,
                'email' => $booking->email,
                'phone' => $booking->phone,
                'gameName' => $booking->gameName,
                'price' => $booking->price,
                'status' => $booking->status,
                'date' => $booking->created_at->toDateString(),
            ];
        });

        return response()->json(['success' => true, 'data' => $bookings]);
    }

    /**
     * Get Key Performance Indicators (KPIs)
     */
    public function kpis(Request $request)
{
    // Apply your filters (like dates or user_id) to the base query here
    $query = Booking::query();
 // Apply date filters if provided
        if ($request->has('startDate') && $request->startDate !== 'all') {
            $query->whereDate('created_at', '>=', $request->startDate);
        }
        if ($request->has('endDate') && $request->endDate !== 'all') {
            $query->whereDate('created_at', '<=', $request->endDate);
        }
    $stats = $query->selectRaw("
        COUNT(*) as total_bookings,
        SUM(quantity) as total_tickets,
        SUM(CASE WHEN status IN ('Admitted', 'Completed') THEN price ELSE 0 END) as total_revenue,
        SUM(CASE WHEN status = 'Admitted' THEN price ELSE 0 END) as net_revenue,
        COUNT(CASE WHEN status = 'Admitted' THEN 1 END) as admitted_count
    ")->first();

    // Calculations using the results from the single query
    $totalBookings = (int) $stats->total_bookings;
    $totalRevenue = (float) $stats->total_revenue;
    $admittedCount = (int) $stats->admitted_count;

    $avgOrderValue = $totalBookings > 0 ? $totalRevenue / $totalBookings : 0;
    $conversionRate = $totalBookings > 0 ? ($admittedCount / $totalBookings) * 100 : 0;

    return response()->json([
        'success' => true,
        'data' => [
            "admittedBookings" => $admittedCount,
            'totalRevenue'    => $totalRevenue,
            'totalNetRevenue' => (float) $stats->net_revenue,
            'totalBookings'   => $totalBookings,
            'totalTickets'    => (int) $stats->total_tickets,
            'avgOrderValue'   => round($avgOrderValue, 2),
            'conversionRate'  => round($conversionRate, 2),
            'admittedCount'   => $admittedCount,
        ]
    ]);
}

    /**
     * Get game-specific analytics
     */
    public function gameAnalytics(Request $request)
    {
        $query = Booking::query();

        // Apply date filters if provided
        if ($request->has('startDate') && $request->startDate !== 'all') {
            $query->whereDate('created_at', '>=', $request->startDate);
        }
        if ($request->has('endDate') && $request->endDate !== 'all') {
            $query->whereDate('created_at', '<=', $request->endDate);
        }

        // Popular Games (bookings count)
        $popularGames = (clone $query)->select('gameName', DB::raw('COUNT(*) as booking_count'))
            ->groupBy('gameName')
            ->orderBy('booking_count', 'desc')
            ->get();

        // Revenue by Game
        $revenueByGame = (clone $query)->select('gameName', DB::raw('SUM(price) as total_revenue'))
            ->groupBy('gameName')
            ->orderBy('total_revenue', 'desc')
            ->get();

        // Capacity Insights (average quantity per booking per game)
        $capacityInsights = (clone $query)->select('gameName', DB::raw('AVG(quantity) as avg_quantity'))
            ->groupBy('gameName')
            ->orderBy('avg_quantity', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'popularGames' => $popularGames,
                'revenueByGame' => $revenueByGame,
                'capacityInsights' => $capacityInsights,
            ]
        ]);
    }

    /**
     * Get financial and payment insights
     */
    public function financialInsights(Request $request)
    {
        $query = Booking::query();

        // Apply date filters if provided
        if ($request->has('startDate') && $request->startDate !== 'all') {
            $query->whereDate('created_at', '>=', $request->startDate);
        }
        if ($request->has('endDate') && $request->endDate !== 'all') {
            $query->whereDate('created_at', '<=', $request->endDate);
        }

        // Payment Status Summary
        $paymentStatusSummary = (clone $query)->select('payment_status_description', DB::raw('COUNT(*) as count'))
            ->whereNotNull('payment_status_description')
            ->groupBy('payment_status_description')
            ->get();

        // Price Point Analysis (group by price ranges)
        $pricePointAnalysis = (clone $query)->select(
                DB::raw('CASE
                    WHEN price < 100 THEN "Under Ksh 100"
                    WHEN price BETWEEN 100 AND 500 THEN "Ksh 100-500"
                    WHEN price BETWEEN 501 AND 1000 THEN "Ksh 501-1000"
                    WHEN price BETWEEN 1001 AND 2000 THEN "Ksh 1001-2000"
                    ELSE "Over Ksh 2000"
                END as price_range'),
                DB::raw('COUNT(*) as booking_count'),
                DB::raw('SUM(price * quantity) as total_revenue')
            )
            ->groupBy('price_range')
            ->orderBy(DB::raw('MIN(price)'), 'asc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'paymentStatusSummary' => $paymentStatusSummary,
                'pricePointAnalysis' => $pricePointAnalysis,
            ]
        ]);
    }

    /**
     * Get time-based trends
     */
    public function timeTrends(Request $request)
    {
        // Booking Volume Over Time (daily for last 30 days)
        $bookingVolume = Booking::select(
                DB::raw('DATE(created_at) as date'),
                DB::raw('COUNT(*) as booking_count'),
                DB::raw('SUM(quantity) as ticket_count')
            )
            ->whereDate('created_at', '>=', Carbon::now()->subDays(30)->toDateString())
            ->groupBy(DB::raw('DATE(created_at)'))
            ->orderBy(DB::raw('DATE(created_at)'), 'asc')
            ->get();

        // Upcoming Traffic (next 30 days by event date)
        $upcomingTraffic = Booking::select(
                'date',
                DB::raw('COUNT(*) as booking_count'),
                DB::raw('SUM(quantity) as ticket_count')
            )
            ->where('date', '>=', Carbon::today()->toDateString())
            ->where('date', '<=', Carbon::today()->addDays(30)->toDateString())
            ->groupBy('date')
            ->orderBy('date', 'asc')
            ->get();

        // Peak Booking Times (by day of week)
        $peakBookingTimes = Booking::select(
                DB::raw('DAYOFWEEK(created_at) as day_of_week'),
                DB::raw('DAYNAME(created_at) as day_name'),
                DB::raw('COUNT(*) as booking_count')
            )
            ->groupBy(DB::raw('DAYOFWEEK(created_at)'), DB::raw('DAYNAME(created_at)'))
            ->orderBy(DB::raw('DAYOFWEEK(created_at)'))
            ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'bookingVolume' => $bookingVolume,
                'upcomingTraffic' => $upcomingTraffic,
                'peakBookingTimes' => $peakBookingTimes,
            ]
        ]);
    }

    /**
     * Get operational details
     */
    public function operationalDetails(Request $request)
    {
        // Recent Bookings (last 10)
        $recentBookings = Booking::orderBy('created_at', 'desc')
            ->limit(10)
            ->get(['ticketId', 'name', 'gameName', 'status', 'created_at']);

        // Customer Contacts (recent bookings)
        $customerContacts = Booking::select('name', 'email', 'phone', 'created_at')
            ->orderBy('created_at', 'desc')
            ->limit(20)
            ->get();

        // Queue Management (bookings with queue positions)
        $queueManagement = Booking::whereNotNull('queuePosition')
            ->orderBy('queuePosition', 'asc')
            ->get(['ticketId', 'name', 'gameName', 'queuePosition', 'status']);

        return response()->json([
            'success' => true,
            'data' => [
                'recentBookings' => $recentBookings,
                'customerContacts' => $customerContacts,
                'queueManagement' => $queueManagement,
            ]
        ]);
    }

    /**
     * Get alerts and warnings
     */
    public function alerts(Request $request)
    {
        // Status Alerts (non-admitted bookings)
        $statusAlerts = Booking::where('status', '!=', 'Admitted')
            ->where('created_at', '>=', Carbon::now()->subDays(7))
            ->orderBy('created_at', 'desc')
            ->get(['ticketId', 'name', 'gameName', 'status', 'created_at']);

        // High-Value Bookings (quantity > 5 or revenue > 5000)
        $highValueBookings = Booking::where(function($query) {
                $query->where('quantity', '>', 5)
                      ->orWhere(DB::raw('price'), '>', 5000);
            })
            ->where('created_at', '>=', Carbon::now()->subDays(30))
            ->orderBy(DB::raw('price'), 'desc')
            ->get(['ticketId', 'name', 'gameName', 'quantity', 'price', 'created_at']);

        return response()->json([
            'success' => true,
            'data' => [
                'statusAlerts' => $statusAlerts,
                'highValueBookings' => $highValueBookings,
            ]
        ]);
    }
  /**
 * Update booking status to 'Admitted' by ticketId.
 */
public function admit(Request $request, $ticketId)
{
    Log::channel('stderr')->info("BookingController::admit called for ticket: $ticketId");

    $booking = Booking::where('ticketId', $ticketId)->first();

    if (!$booking) {
        return response()->json([
            'success' => false,
            'message' => "Ticket ID {$ticketId} not found."
        ], 404);
    }

    // Optional: Check if already admitted to prevent duplicate scans
    if ($booking->status === 'Admitted') {
        return response()->json([
            'success' => true, 
            'message' => 'This ticket has already been admitted.',
            'already_admitted' => true,
            'data' => $booking
        ], 200);
    }

    $booking->update(['status' => 'Admitted']);

    // Log admin activity
    AdminActivityLog::create([
        'user_id' => auth()->id(),
        'action' => 'admitted_ticket',
        'details' => [
            'ticket_id' => $ticketId,
            'booking_id' => $booking->id,
            'previous_status' => $booking->getOriginal('status'),
            'new_status' => 'Admitted',
        ],
        'ip_address' => $request->ip(),
        'user_agent' => $request->userAgent(),
    ]);

    return response()->json([
        'success' => true,
        'message' => 'Ticket successfully admitted.',
        'data' => $booking
    ], 200);
}
}
