<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use App\Mail\EmailVerificationMail;
use App\Mail\PasswordResetMail;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;



class AuthController extends Controller
{
    /**
     * Register a new user.
     */
    public function signup(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'email' => 'required|string|email|max:255|unique:users',
                'password' => 'required|string|min:8|confirmed',
            ]);

            if ($validator->fails()) {
                $errors = $validator->errors();
                if ($errors->has('email') && $errors->get('email')[0] === 'The email has already been taken.') {
                    return response()->json(['message' => 'The email has already been taken.'], 409);
                }
                return response()->json($validator->errors(), 422);
            }

            $userRole = Role::where('name', 'User')->first();
            if (!$userRole) {
                return response()->json(['message' => 'Default "User" role not found. Please run database seeders.'], 500);
            }

            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'role_id' => $userRole->id,
                'status' => 'inactive',
            ]);

            $verification_token = Str::random(60);

            DB::table('user_verification_tokens')->insert([
                'user_id' => $user->id,
                'token' => $verification_token,
                'created_at' => now(),
            ]);

            // Send verification email
            $verification_url = config('app.url') . '/verify-email?token=' . $verification_token;
            Mail::to($user->email)->send(new EmailVerificationMail($verification_url));

            $token = $user->createToken('auth_token')->plainTextToken;

            return response()->json([
                'access_token' => $token,
                'token_type' => 'Bearer',
            ], 201);
        } catch (ValidationException $e) {
            return response()->json(['message' => $e->getMessage()], 422);
        }
    }

    /**
     * Verify a user\'s email address.
     */
    public function verifyEmail(Request $request)
    {
        $token = $request->query('token');

        $verification_token = DB::table('user_verification_tokens')->where('token', $token)->first();

        if (!$verification_token || now()->subMinutes(60)->gt($verification_token->created_at)) {
            return response()->json(['message' => 'Invalid or expired verification token.'], 400);
        }

        $user = User::findOrFail($verification_token->user_id);

        if ($user->status === 'active') {
             return response()->json(['message' => 'This account has already been activated.'], 422);
        }

        $user->status = 'active';
        $user->save();

        DB::table('user_verification_tokens')->where('token', $token)->delete();

        return response()->json(['message' => 'Your email has been verified successfully.']);
    }



public function login(Request $request)
{
    $validator = Validator::make($request->all(), [
        'email' => 'required|email',
        'password' => 'required|string',
    ]);

    if ($validator->fails()) {
        return response()->json([
            'errors' => $validator->errors()
        ], 422);
    }

    if (!Auth::attempt($request->only('email', 'password'))) {
        return response()->json([
            'message' => 'Invalid login details'
        ], 401);
    }

    /**
     * Load user with role + permissions
     * This is REQUIRED for the permissions accessor to work
     */
    $user = User::with('role.permissions')
        ->where('email', $request->email)
        ->firstOrFail();

    if ($user->status !== 'active') {
        return response()->json([
            'message' => 'Account not verified.'
        ], 403);
    }

    $token = $user->createToken('auth_token')->plainTextToken;

    return response()->json([
        'access_token' => $token,
        'token_type'   => 'Bearer',
        'user'         => $user, // includes permissions via accessor
    ]);
}


    /**
     * Resend the verification email.
     */
    public function resendVerificationEmail(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email|exists:users,email',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $user = User::where('email', $request->email)->first();

        if ($user->status === 'active') {
            return response()->json(['message' => 'This account has already been activated.'], 422);
        }

        // Generate a new verification token
        $verification_token = Str::random(60);

        DB::table('user_verification_tokens')->updateOrInsert(
            ['user_id' => $user->id],
            [
                'token' => $verification_token,
                'created_at' => now(),
            ]
        );

        $verification_url = config('app.url') . '/verify-email?token=' . $verification_token;
        Mail::to($user->email)->send(new EmailVerificationMail($verification_url));

        return response()->json(['message' => 'A new verification link has been sent to your email address.']);
    }

    /**
     * Send a password reset code.
     */
    public function forgotPassword(Request $request)
    {
        $request->validate(['email' => 'required|email|exists:users,email']);

        $reset_code = random_int(100000, 999999);

        DB::table('password_reset_tokens')->updateOrInsert(
            ['email' => $request->email],
            [
                'token' => Hash::make($reset_code),
                'created_at' => now()
            ]
        );
        
        Mail::to($request->email)->send(new PasswordResetMail($reset_code));

        return response()->json([
            'message' => 'A password reset code has been sent to your email.',
        ]);
    }

    /**
     * Reset the user\'s password using a reset code.
     */
    public function resetPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email|exists:users,email',
            'token' => 'required|string|min:6|max:6',
            'password' => 'required|string|min:8|confirmed',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $passwordReset = DB::table('password_reset_tokens')
            ->where('email', $request->email)
            ->first();

        if (!$passwordReset || !Hash::check($request->token, $passwordReset->token) || now()->subMinutes(10)->gt($passwordReset->created_at)) {
            return response()->json(['message' => 'Invalid or expired password reset token.'], 400);
        }

        $user = User::where('email', $request->email)->firstOrFail();
        $user->password = Hash::make($request->password);
        $user->save();

        DB::table('password_reset_tokens')->where('email', $request->email)->delete();

        return response()->json(['message' => 'Your password has been reset successfully.']);
    }
}
